<?php
require_once('utility.php');
function list_all_esxi_vmx_file_usage() {
    $filename = basename( __FILE__, ".inc" ) ;
    
    echo "Usage: $filename <Host> <UserName> <Password> [Conditions]\n" ;
    echo "  <EsxiHost>     mandatory: ESXI host ip\n" ;
    echo "  <EsxiUserName> mandatory: ESXI valid username\n" ;
    echo "  <EsxiPassword> mandatory: ESXI valid password\n" ;
    echo "  [Subject]      optional : list out specific vmx files\n" ;
    echo "      ManagementServer\n" ;
    echo "      SandboxController\n" ;
    echo "      Sandbox\n" ;
    echo "      All(default)\n" ;
    echo "\n" ;
    echo "  [Conditions]   optional : only for subject 'Sandbox', to filter out Sandbox vmx files\n" ;
    echo "      New    : list vmx files which are newly imported by user\n" ;
    echo "      Origin : list vmx files which are backup before sandcastle is installed\n" ;
    echo "      Scas   : list vmx files sandcastle is installed\n" ;
    echo "      Cloned : list vmx files which are cloned" ;
    echo "\n" ;
    echo "Example: query all vmx file\n" ;
    echo "    php backend.php $filename <EsxiHost> <EsxiUserName> <EsxiPassword> All\n" ;
    echo "\n" ;
    echo "Example: query ManagementServer vmx file\n" ;
    echo "    php backend.php $filename <EsxiHost> <EsxiUserName> <EsxiPassword> ManagementServer\n" ;
    echo "\n" ;
    echo "Example: query Original Sandbox vmx files\n" ;
    echo "    php backend.php $filename <EsxiHost> <EsxiUserName> <EsxiPassword> Sandbox Origin\n" ;
    echo "\n" ;
}

function list_all_esxi_vmx_file_filtering_by_prefix( $vmxFilePathArray, $prefix ) {
    $result = array() ;
    foreach( $vmxFilePathArray as $path ) {
        $folderName = basename( dirname( $path ) ) ;
        $baseName   = basename( $path, ".vmx" ) ;
        
        # case-insensitive
        $isFolderNameMatched = ( stripos( $folderName, $prefix, 0 ) === 0 ) ;
        $isBaseNameMatched   = ( stripos( $baseName,   $prefix, 0 ) === 0 ) ;
        
        if( $isFolderNameMatched == true && $isBaseNameMatched == true ) {
            array_push( $result, $path ) ;
        }
    }
    return $result ;
}

function list_all_esxi_vmx_file_check_sb_type( $vmxFilePath, $esxiHost, $esxiUsername, $esxiPassword ) {
    $folderName = dirname( $vmxFilePath ) ;
    
    ############################################################
    # check if it is "ORIGIN", "SCAS", "CLONED", or "New"
    ############################################################
    
    # for performance concern, merge 3 commands into one command
    $resultArray = run_command_at_esxi( array(
        'exit_when_error' => 0,
        'command' => "/bin/find $folderName -name ORIGIN;" .
                     "/bin/find $folderName -name CLONED;" .
                     "/bin/find $folderName -name SCAS"
    ) ) ;
    $resultValue    = $resultArray[ 0 ] ;
    $resultMsgArray = $resultArray[ 1 ] ;
    if( $resultValue !== 0 ) {
        throw new Exception( join( "\n", $resultMsgArray ) ) ;
    }

    # file "ORIGIN" found (case-sensitive)
    if( in_array( "$folderName/ORIGIN", $resultMsgArray, true ) == true ) {
        return "origin" ;
    }
    
    # file "SCAS" found (case-sensitive)
    elseif( in_array( "$folderName/SCAS", $resultMsgArray, true ) == true ) {
        return "scas" ;
    }
    
    # file "CLONED" found (case-sensitive)
    elseif( in_array( "$folderName/CLONED", $resultMsgArray, true ) == true ) {
        return "cloned" ;
    }
    
    # otherwise, it is sandbox image user just imported
    else {
        return "new" ;
    }
}

function list_all_esxi_vmx_file_filtering_by_condition( $vmxFilePathArray, $condition, $esxiHost, $esxiUsername, $esxiPassword ) {
    if( $condition == "none" ) {
        return $vmxFilePathArray ;
    }
    
    $originArray    = array() ;
    $scasArray      = array() ;
    $clonedArray    = array() ;
    $newArray       = array() ;
    $availableArray = array() ;

    foreach( $vmxFilePathArray as $path ) {
        $sbType = list_all_esxi_vmx_file_check_sb_type(
            $path,
            $esxiHost,
            $esxiUsername,
            $esxiPassword ) ;

        switch( $sbType ) {
            case "origin" :
                $path = str_ireplace( "_ORIGIN", "", $path ) ;
                array_push( $originArray, $path ) ;
                array_push( $availableArray, $path ) ;
                break ;
            
            case "scas" :
                array_push( $scasArray, $path ) ;
                break ;
            
            case "cloned" :
                array_push( $clonedArray, $path ) ;
                break ;
                
            case "new" :
                array_push( $newArray, $path ) ;
                array_push( $availableArray, $path ) ;
                break ;
                
            default :
                throw new Exception( "UnsupportSandboxType: sbType='$sbType'" ) ;
        }
    }
    
    switch( $condition ) {
        case "new" :
            return $newArray ;
            
        case "origin" :
            return $originArray ;
            
        case "scas" :
            return $scasArray ;
            
        case "cloned" :
            return $clonedArray ;
            
        case "available" :
            return $availableArray ;

        default:
            throw new Exception( "UnsupportSandboxCondition: sbCondition='$condition'" ) ;
    }
}

function list_all_esxi_vmx_file( $argv ) {
    global $DTASCFG_XML_FILE ;
    
	# check number of arguments
    if( count( $argv ) < 3 || count( $argv ) > 5 ) {
		list_all_esxi_vmx_file_usage() ;
		exit( 1 ) ;
    }
    
    # get prefix info
    $xml = simplexml_load_file( $DTASCFG_XML_FILE ) ;
    $ms_prefix = chop( $xml -> management_server -> prefix ) ;
    $sc_prefix = chop( $xml -> sandbox_controller -> prefix ) ;
    $sb_prefix = chop( $xml -> sandbox -> prefix ) ;	
    
    # get esxi info
    $esxi_host     = array_shift( $argv ) ;
    $esxi_username = array_shift( $argv ) ;
    $esxi_password = array_shift( $argv ) ;
    
    # get subject and assign "all" in case it is not given
    $subject = array_shift( $argv ) ;
    $subject = ($subject == null) ?
        "all" :
        strtolower( $subject ) ;
        
    # get condition and assign "none" in case it is not given
    $condition = array_shift( $argv ) ;
    $condition = ($condition == null) ?
        "none" :
        strtolower( $condition ) ;

    $retValue = 0 ;
    $result = array() ;
    try {
        $portNumber = getEsxiPort($esxi_host);
        # Firstly, get all vmx files from ESXi
        $resultArray = run_program_at_ms( array(
            'exit_when_error' => 0,
            'program'   => "/usr/bin/vmware-cmd",
            'arguments' => array(
                    '--server',   $esxi_host,
                    '--portnumber', $portNumber,
                    '--username', $esxi_username,
                    '--password', $esxi_password,
                    '-l'
        ) ) ) ;
        $resultValue    = $resultArray[ 0 ] ;
        $resultMsgArray = $resultArray[ 1 ] ;
        if( $resultValue !== 0 ) {
            $retValue = $resultValue ;
            throw new Exception( join( "\n", $resultMsgArray ) ) ;
        }
        $vmxFilePathArray = $resultMsgArray ;

        # main logic
        switch ( $subject ) {
            case "all" :
                $result = $vmxFilePathArray ;
                break ;
            
            case strtolower( $ms_prefix ) :
            case "managementserver" :
            case "ms" :
                $result = list_all_esxi_vmx_file_filtering_by_prefix(
                    $vmxFilePathArray,
                    $ms_prefix ) ;
                    
                break ;
                
            case strtolower( $sc_prefix ) :
            case "sandboxcontroller" :
            case "sc" :
                $result = list_all_esxi_vmx_file_filtering_by_prefix(
                    $vmxFilePathArray,
                    $sc_prefix ) ;

                break ;
                
            case strtolower( $sb_prefix ) :
            case "sandbox" :
            case "sb" :
                $allVmxFileWithSbPrefix = list_all_esxi_vmx_file_filtering_by_prefix(
                    $vmxFilePathArray,
                    $sb_prefix ) ;

                $result = list_all_esxi_vmx_file_filtering_by_condition(
                    $allVmxFileWithSbPrefix,
                    $condition,
                    $esxi_host,
                    $esxi_username,
                    $esxi_password ) ;

                break ;
            
            default :
                $result = list_all_esxi_vmx_file_filtering_by_prefix(
                    $vmxFilePathArray,
                    $subject ) ;
            
                break ;
        }
    }
    
    catch( Exception $e ) {
        $msg = "[" . $e -> getFile() . "(" . $e -> getLine() . ")]: " . $e -> getMessage() ;
        
        debug_print( "Exception caught: msg='$msg'" ) ;
        echo $msg . "\n" ;
        exit( $retValue ) ;
    }
    
    echo join( "\n", $result ) . "\n" ;
    exit( $retValue ) ;
}

